/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.transactions.cleanup;

import com.couchbase.client.core.cnc.Event;
import com.couchbase.client.core.config.ClusterConfig;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.retry.reactor.Retry;
import com.couchbase.client.java.ReactiveCollection;
import com.couchbase.transactions.atr.ATRIds;
import com.couchbase.transactions.cleanup.ATRStats;
import com.couchbase.transactions.cleanup.ClientRecord;
import com.couchbase.transactions.cleanup.ClientRecordDetails;
import com.couchbase.transactions.cleanup.ClusterData;
import com.couchbase.transactions.cleanup.TransactionsCleanup;
import com.couchbase.transactions.components.ATREntry;
import com.couchbase.transactions.components.ATRUtil;
import com.couchbase.transactions.components.ActiveTransactionRecord;
import com.couchbase.transactions.config.TransactionConfig;
import com.couchbase.transactions.error.internal.ThreadStopRequested;
import com.couchbase.transactions.log.LostCleanupThreadEndedPrematurely;
import com.couchbase.transactions.log.SimpleEventBusLogger;
import com.couchbase.transactions.log.TransactionCleanupAttempt;
import com.couchbase.transactions.log.TransactionCleanupEndRunEvent;
import com.couchbase.transactions.log.TransactionCleanupStartRunEvent;
import com.couchbase.transactions.support.SpanWrapper;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;

public class LostCleanupDistributed {
    private final ClientRecord clientRecord;
    private final SimpleEventBusLogger LOGGER;
    private final ClusterData clusterData;
    private final TransactionConfig config;
    private final TransactionsCleanup parent;
    private volatile boolean stop = false;
    private CountDownLatch stopLatch;
    private final Duration actualCleanupWindow;
    private final String clientUuid = UUID.randomUUID().toString();
    private final String bp;
    private final Set<String> bucketThreads = ConcurrentHashMap.newKeySet();

    public LostCleanupDistributed(TransactionConfig config, ClusterData clusterData, TransactionsCleanup parent) {
        this.LOGGER = new SimpleEventBusLogger(clusterData.cluster().environment().eventBus(), TransactionsCleanup.LOST_CATEGORY);
        this.clientRecord = config.clientRecordFactory().create(config, clusterData);
        this.clusterData = clusterData;
        this.config = config;
        this.parent = parent;
        this.actualCleanupWindow = config.cleanupWindow();
        this.bp = String.format("Client %s", this.clientUuid.substring(0, 5));
    }

    public String clientUuid() {
        return this.clientUuid;
    }

    public void stop() {
        this.stopThreads();
        this.clientRecord.removeClientFromAllBuckets(this.clientUuid).onErrorResume(err -> {
            this.LOGGER.warn(String.format("%s failed to remove from all buckets with err: %s", this.bp, err));
            return Mono.empty();
        }).blockLast();
        this.LOGGER.info(String.format("%s stopped lost cleanup process", this.bp));
    }

    public void stopThreads() {
        int numBucketThreads = this.bucketThreads.size();
        this.LOGGER.info(String.format("%s stopping lost cleanup process, waiting for %d threads to end", this.bp, numBucketThreads));
        this.stopLatch = new CountDownLatch(numBucketThreads);
        this.stop = true;
        try {
            if (!this.stopLatch.await(10L, TimeUnit.SECONDS)) {
                this.LOGGER.info("Lost background cleanup threads did not stop in time");
            }
        }
        catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    private boolean logDebug() {
        return true;
    }

    private boolean logVerbose() {
        return true;
    }

    public static List<String> atrsToHandle(int indexOfThisClient, int numActiveClients, int numAtrs) {
        List<String> allAtrs = ATRIds.allAtrs(numAtrs);
        ArrayList<String> out = new ArrayList<String>();
        for (int i = indexOfThisClient; i < allAtrs.size(); i += numActiveClients) {
            out.add(allAtrs.get(i));
        }
        return out;
    }

    private Flux<TransactionCleanupAttempt> handleATRCleanup(String bp, ReactiveCollection atrCollection, String atrId, ATRStats stats) {
        return Flux.defer(() -> {
            long start = System.nanoTime();
            AtomicLong timeToFetchAtr = new AtomicLong(0L);
            SpanWrapper span = SpanWrapper.create(this.config, "transaction_distclean_atr").start();
            Duration SAFETY_MARGIN = Duration.of(1500L, ChronoUnit.MILLIS);
            return this.parent.getCleaner().beforeAtrGet(atrId).then(ActiveTransactionRecord.getAndTouchAtr(atrCollection, atrId, this.config.keyValueTimeout(), span, this.config)).flatMap(v -> {
                timeToFetchAtr.set(System.nanoTime());
                if (v.isPresent()) {
                    return Mono.just(v.get());
                }
                return Mono.empty();
            }).doOnError(err -> {
                this.LOGGER.info(String.format("%s Got error '%s' while getting ATR %s/", bp, err, ATRUtil.getAtrDebug(atrCollection, atrId)));
                stats.errored = Optional.of(err);
            }).flatMapMany(atr -> {
                Collection expired;
                stats.numEntries = atr.entries().size();
                stats.exists = true;
                stats.errored = Optional.empty();
                stats.expired = expired = (Collection)atr.entries().stream().filter(v -> v.hasExpired(SAFETY_MARGIN.toMillis(), this.config.transactionExpirationTime())).collect(Collectors.toList());
                return Flux.fromIterable((Iterable)expired);
            }).concatMap(atrEntry -> {
                if (this.logDebug()) {
                    this.LOGGER.verbose(String.format("%s Found expired attempt %s, expires after %d, age %d (started %d, now %d)", bp, atrEntry.attemptId(), atrEntry.expiresAfterMsecs().orElse(-1), atrEntry.ageMsecs(), atrEntry.timestampStartMsecs().orElse(0L), atrEntry.cas() / 1000000L));
                }
                stats.expiredEntryCleanupTotalAttempts.incrementAndGet();
                return this.parent.getCleaner().cleanupATREntry(atrCollection, atrId, atrEntry.attemptId(), (ATREntry)atrEntry, false).onErrorResume(err -> {
                    stats.expiredEntryCleanupFailedAttempts.incrementAndGet();
                    return Mono.empty();
                });
            }).doFinally(v -> {
                long now = System.nanoTime();
                this.LOGGER.verbose(String.format("%s processed ATR %s after %d millis (%d fetching ATR): %s", bp, ATRUtil.getAtrDebug(atrCollection, atrId), (now - start) / 1000000L, (timeToFetchAtr.get() - start) / 1000000L, stats));
                span.finish();
            });
        });
    }

    void configureFromClusterConfig(ClusterConfig config) {
        if (this.stop) {
            this.LOGGER.info(String.format("ignoring new cluster config with %d buckets as stopping", config.bucketConfigs().size()));
        } else {
            this.LOGGER.info(String.format("new cluster config with %d buckets", config.bucketConfigs().size()));
            Flux threads = Flux.fromIterable(config.bucketConfigs().values()).flatMap(bc -> {
                String bucketName = bc.name();
                boolean wasNotTracking = this.bucketThreads.add(bucketName);
                if (wasNotTracking) {
                    this.LOGGER.info(String.format("will start cleaning lost transactions on bucket %s", RedactableArgument.redactMeta((Object)bucketName)));
                    return this.clusterData.getBucketFromName(bucketName).waitUntilReady(Duration.ofSeconds(Integer.MAX_VALUE)).then(this.clusterData.getBucketDefaultCollection(bucketName)).flatMap(collection -> {
                        this.LOGGER.info(String.format("%s %s/%s creating thread to handle lost transactions", this.bp, RedactableArgument.redactMeta((Object)collection.bucketName()), RedactableArgument.redactMeta((Object)collection.name())));
                        return this.perBucketThread((ReactiveCollection)collection).doOnError(err -> {
                            if (!(err instanceof ThreadStopRequested)) {
                                this.LOGGER.warn(String.format("%s %s/%s lost transactions thread has ended on error %s", this.bp, RedactableArgument.redactMeta((Object)collection.bucketName()), RedactableArgument.redactMeta((Object)collection.name()), err));
                            }
                        });
                    });
                }
                return Mono.empty();
            });
            threads.subscribe(v -> this.LOGGER.warn(String.format("%s lost transactions cleanup thread(s) ending", this.bp)), err -> {
                if (!(err instanceof ThreadStopRequested)) {
                    this.LOGGER.warn(String.format("%s lost transactions cleanup thread ended with exception " + err, this.bp));
                    this.clusterData.cluster().environment().eventBus().publish((Event)new LostCleanupThreadEndedPrematurely((Throwable)err));
                }
            });
        }
    }

    void start() {
        this.configureFromClusterConfig(this.clusterData.cluster().core().clusterConfig());
        this.clusterData.cluster().core().configurationProvider().configs().subscribe(this::configureFromClusterConfig);
    }

    private Mono<Void> perBucketThread(ReactiveCollection collection) {
        return Mono.defer(() -> {
            String bp = "lost/" + collection.bucketName() + "/" + collection.name() + "/clientId=" + this.clientUuid.substring(0, 5);
            return this.clientRecord.processClient(this.clientUuid, collection, this.config).flatMap(clientDetails -> {
                long startOfRun = System.nanoTime();
                HashMap atrStats = new HashMap();
                List<String> atrsHandledByThisClient = LostCleanupDistributed.atrsToHandle(clientDetails.indexOfThisClient(), clientDetails.numActiveClients(), this.config.numAtrs());
                long checkAtrEveryNMillis = Math.max(1L, this.actualCleanupWindow.toMillis() / (long)atrsHandledByThisClient.size());
                if (atrsHandledByThisClient.size() < this.config.numAtrs()) {
                    atrsHandledByThisClient.forEach(id -> this.LOGGER.verbose(String.format("%s owns ATR %s (of %d) and will check it over next %dmillis, checking an ATR every %dmillis", bp, id, this.config.numAtrs(), this.actualCleanupWindow.toMillis(), checkAtrEveryNMillis)));
                } else {
                    this.LOGGER.verbose(String.format("%s owns all %d ATRs and will check them over next %dmills, checking an ATR every %dmillis", bp, this.config.numAtrs(), this.actualCleanupWindow.toMillis(), checkAtrEveryNMillis));
                }
                TransactionCleanupStartRunEvent ev = new TransactionCleanupStartRunEvent(collection.bucketName(), collection.name(), this.clientUuid, (ClientRecordDetails)clientDetails, this.actualCleanupWindow, atrsHandledByThisClient.size(), this.config.numAtrs(), Duration.ofMillis(checkAtrEveryNMillis));
                this.clusterData.cluster().environment().eventBus().publish((Event)ev);
                return Flux.fromIterable(atrsHandledByThisClient).delayElements(Duration.of(checkAtrEveryNMillis, ChronoUnit.MILLIS)).concatMap(atrId -> {
                    if (this.logVerbose()) {
                        this.LOGGER.verbose(String.format("%s checking for lost txns in atr %s", bp, ATRUtil.getAtrDebug(collection, atrId)));
                    }
                    ATRStats stats = new ATRStats();
                    return this.checkIfThreadStopped(collection, (String)atrId).thenMany(this.handleATRCleanup(bp, collection, (String)atrId, stats)).then(Mono.fromRunnable(() -> atrStats.put(atrId, stats))).thenReturn(atrId);
                }).onErrorResume(err -> {
                    if (err instanceof ThreadStopRequested) {
                        return Mono.error((Throwable)err);
                    }
                    this.LOGGER.debug(String.format("%s lost cleanup thread got error '%s', continuing", bp, err));
                    return Mono.empty();
                }).then().thenReturn((Object)Tuples.of(atrStats, (Object)((Object)ev), (Object)startOfRun));
            }).doOnNext(stats -> {
                Duration timeForRun = Duration.ofNanos(System.nanoTime() - (Long)stats.getT3());
                TransactionCleanupEndRunEvent ev = new TransactionCleanupEndRunEvent((TransactionCleanupStartRunEvent)((Object)((Object)((Object)stats.getT2()))), (Map)stats.getT1(), timeForRun);
                this.clusterData.cluster().environment().eventBus().publish((Event)ev);
            }).retryWhen((Function)Retry.allBut((Class[])new Class[]{ThreadStopRequested.class}).exponentialBackoff(Duration.of(100L, ChronoUnit.MILLIS), Duration.of(2000L, ChronoUnit.MILLIS)).doOnRetry(v -> this.LOGGER.info(String.format("%s retrying lost cleanup on error '%s'", bp, v.exception()))).retryMax(100000L)).repeat().then();
        });
    }

    private Mono<String> checkIfThreadStopped(ReactiveCollection collection, String v) {
        return Mono.defer(() -> {
            if (this.stop) {
                this.LOGGER.info(String.format("Stopping background cleanup thread for lost transactions on %s/%s", collection.bucketName(), collection.name()));
                this.stopLatch.countDown();
                return Mono.error((Throwable)new ThreadStopRequested());
            }
            return Mono.just((Object)v);
        });
    }
}

